<?php
/**
 * Add widget all-items to Elementor
 *
 * @since   1.3.13
 */

class Havezic_BABE_Elementor_Itemfeature_Widget extends \Elementor\Widget_Base {

    /**
     * Get widget name.
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'babe-item-feature';
    }

    /**
     * Get widget title.
     *
     * @return string Widget title.
     */
    public function get_title() {
        return esc_html__('Item feature', 'havezic');
    }

    /**
     * Get widget icon.
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'eicon-toggle';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @return array Widget keywords.
     */
    public function get_keywords() {
        return ['feature'];
    }

    /**
     * Get widget categories.
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return ['book-everything-elements'];
    }

    /**
     * Register widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     */
    protected function register_controls() {

        $this->title_controls();
        $this->add_control_style_wrapper();

    }

    protected function title_controls() {
        $this->start_controls_section(
            'section_title',
            [
                'label' => __('Title Setting', 'havezic'),
            ]
        );

        $this->add_control(
            'enable_title',
            [
                'label'   => esc_html__('Enable title', 'havezic'),
                'type'    => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'title',
            [
                'label'       => __('Title', 'havezic'),
                'type'        => \Elementor\Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
                'placeholder' => __('Enter your title', 'havezic'),
                'default'     => __('What’s Included', 'havezic'),
                'condition'   => [
                    'enable_title' => 'yes',
                ]
            ]
        );


        $this->add_control(
            'size',
            [
                'label'     => __('Size', 'havezic'),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'default'   => 'default',
                'options'   => [
                    'default' => __('Default', 'havezic'),
                    'small'   => __('Small', 'havezic'),
                    'medium'  => __('Medium', 'havezic'),
                    'large'   => __('Large', 'havezic'),
                    'xl'      => __('XL', 'havezic'),
                    'xxl'     => __('XXL', 'havezic'),
                ],
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'header_size',
            [
                'label'     => __('HTML Tag', 'havezic'),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'options'   => [
                    'h1'   => 'H1',
                    'h2'   => 'H2',
                    'h3'   => 'H3',
                    'h4'   => 'H4',
                    'h5'   => 'H5',
                    'h6'   => 'H6',
                    'div'  => 'div',
                    'span' => 'span',
                    'p'    => 'p',
                ],
                'default'   => 'h2',
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label'     => __('Alignment', 'havezic'),
                'type'      => \Elementor\Controls_Manager::CHOOSE,
                'options'   => [
                    'left'    => [
                        'title' => __('Left', 'havezic'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => __('Center', 'havezic'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => __('Right', 'havezic'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => __('Justified', 'havezic'),
                        'icon'  => 'eicon-text-align-justify',
                    ],
                ],
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_title_style',
            [
                'label'     => __('Title', 'havezic'),
                'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => __('Text Color', 'havezic'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-heading-title' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name'      => 'typography',
                'selector'  => '{{WRAPPER}} .elementor-heading-title',
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label'     => __('Margin bottom', 'havezic'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-widget-container .elementor-heading-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'name'      => 'text_shadow',
                'selector'  => '{{WRAPPER}} .elementor-heading-title',
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'blend_mode',
            [
                'label'     => __('Blend Mode', 'havezic'),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'options'   => [
                    ''            => __('Normal', 'havezic'),
                    'multiply'    => 'Multiply',
                    'screen'      => 'Screen',
                    'overlay'     => 'Overlay',
                    'darken'      => 'Darken',
                    'lighten'     => 'Lighten',
                    'color-dodge' => 'Color Dodge',
                    'saturation'  => 'Saturation',
                    'color'       => 'Color',
                    'difference'  => 'Difference',
                    'exclusion'   => 'Exclusion',
                    'hue'         => 'Hue',
                    'luminosity'  => 'Luminosity',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-heading-title' => 'mix-blend-mode: {{VALUE}}',
                ],
                'separator' => 'none',
                'condition' => [
                    'enable_title' => 'yes',
                ]
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     */
    protected function render() {
        $settings     = $this->get_settings_for_display();
        $exclude_html = '';
        ?>
        <div class="elementor-widget-inner">
            <?php $this->title_render();

            if (\Elementor\Plugin::instance()->editor->is_edit_mode()) {
                $post_id = havezic_ba_get_default_single_id();
            } else {
                $post_id = get_the_ID();
            }

            $babe_post = get_post($post_id);

            if (is_single() && $babe_post->post_type == BABE_Post_types::$booking_obj_post_type) {

                $feature = BABE_Post_types::$attr_tax_pref . 'features';

                $terms_active = get_the_terms($post_id, $feature);
                $ids_active   = [];
                if (!empty($terms_active) && !is_wp_error($terms_active)) {
                    foreach ($terms_active as $term) {
                        $ids_active[] = $term->term_id;
                    }
                }
                $terms = get_terms($feature, array(
                    'hide_empty' => false,
                ));

                if (!empty($terms) && !is_wp_error($terms)) {
                    foreach ($terms as $term) {
                        $class = in_array($term->term_id, $ids_active) ? 'active' : '';
                        if ($class == 'active') {
                            ?>
                            <div class="item-feature <?php echo esc_attr($class); ?>">
                                <div class="name-feature <?php echo esc_attr($class); ?>" title="<?php echo esc_attr($term->name); ?>">
                                    <i class="havezic-icon-check"></i>
                                    <span class="title"><?php echo esc_html($term->name); ?></span>
                                </div>
                            </div>
                            <?php
                        } else {
                            ob_start();
                            ?>
                            <div class="item-feature <?php echo esc_attr($class); ?>">
                                <div class="name-feature <?php echo esc_attr($class); ?>" title="<?php echo esc_attr($term->name); ?>">
                                    <i class="havezic-icon-times"></i>
                                    <span class="title"><?php echo esc_html($term->name); ?></span>
                                </div>
                            </div>
                            <?php
                            $exclude_html .= ob_get_clean();
                        }
                        ?>

                    <?php }
                    printf('%s',$exclude_html);
                }
            }
            ?>
        </div>
        <?php
    }

    protected function title_render() {
        $settings = $this->get_settings_for_display();

        if ('' === $settings['title'] || $settings['enable_title'] === 'no') {
            return;
        }

        $this->add_render_attribute('title', 'class', 'elementor-heading-title');

        if (!empty($settings['size'])) {
            $this->add_render_attribute('title', 'class', 'elementor-size-' . $settings['size']);
        }

        $this->add_inline_editing_attributes('title');

        $title = $settings['title'];

        echo sprintf('<%1$s %2$s>%3$s</%1$s>', \Elementor\Utils::validate_html_tag($settings['header_size']), $this->get_render_attribute_string('title'), $title);
    }

    protected function add_control_style_wrapper($condition = array()) {
        $this->start_controls_section(
            'section_style_wrapper',
            [
                'label' => esc_html__('Wrapper', 'havezic'),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_control(
            'wrapper_border_hover',
            [
                'label'     => esc_html__('Border Hover Color', 'havezic'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-widget-inner:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'wrapper_radius',
            [
                'label'      => esc_html__('Border Radius', 'havezic'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-widget-inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'wrapper_box_shadow',
                'selector' => '{{WRAPPER}} .elementor-widget-inner',
            ]
        );

        $this->end_controls_section();
    }

}

$widgets_manager->register(new Havezic_BABE_Elementor_Itemfeature_Widget());
